import asyncio
import time
import serial
import os
import sys
import socket
import fnmatch
from fleming.common.firmware_util import *
from urpc_enum.dualmoverparameter import DualMoverParameter

from py_pli.pylib import VUnits
from virtualunits.HAL import HAL
from virtualunits.vu_scan_table import VUScanTable


hal: HAL = VUnits.instance.hal
st_unit: VUScanTable = hal.scan_table
st_endpoint = get_dual_mover_endpoint('st')

# Scantable Limits (with encoder modifications)
MAX_X = 130.0
MAX_Y = 310.0
# 1536 Well Positions
# FBD Top
A1_X = 71.96
A1_Y = 162.16
AF1_Y = 92.41
AF48__X = 177.71 # ==> Exceed maximun range!!

def get_datafile(test_function):
# Returns the next datafile name for a test function
# Example: test_function = 'test' -> test_Run001.csv
# Checked 2025-05-20/Kay Struebing on Base Tester
    path, script = os.path.split(__file__)
    base_name = script.split('.')[0]
    instrument = socket.gethostname()
    today = datetime.now().strftime('%Y%m%d')
    datapath = f'{path}/{instrument}_{today}'
    if not os.path.isdir(datapath):
        os.mkdir(datapath)
    datafile = f'{base_name}.{test_function}_Run'
    no = len(fnmatch.filter(os.listdir(datapath), datafile + '*.csv')) + 1
    datafile = f'{datafile}{no:03d}.csv'
    return(f'{datapath}/{datafile}')

# Mitutoyo Sensor InterfaceFunctions
def setup_serial_mitutoyo():
    # Serial Config for Mitutoyo Messfühler
    # Checked 2025-05-20/Kay Struebing on Base Tester
    SERIAL_PORT = '/dev/ttyUSB0'
    BAUDRATE = 9600
    BYTESIZE = 7
    PARITY = serial.PARITY_EVEN
    STOPBITS = serial.STOPBITS_TWO
    serial_port = serial.Serial(port = SERIAL_PORT, baudrate=BAUDRATE, bytesize=BYTESIZE, timeout=2, parity=PARITY, stopbits=STOPBITS)
    return (serial_port)

def read_value(sensor):
    sensor.write(b'GA01\r\n') # Read display value
    value = float(sensor.read(size=20)[5:15])
    return(value)

def current_value(sensor):
    sensor.write(b'CN01\r\n') # Set display to current value
    return(1)

def max_value(sensor):
    sensor.write(b'CX01\r\n') #set display to max value
    return(1)

def set_zero(sensor):
    # Checked 2025-05-20/Kay Struebing on Base Tester
    sensor.write(b'CR01\r\n') #set display to zero
    response = sensor.read(size=20)
    return(response)

async def log_sensor(count = 1):
    # Read Sensor-Value and write to file
    # Checked 2025-05-20/Kay Struebing on Base Tester
    test = sys._getframe().f_code.co_name  # get function name from sys
    file_name = get_datafile(test)
    sensor = setup_serial_mitutoyo()
    for i in range(count):
        value = read_value(sensor)
        with open(file_name, 'a') as data:
            print(f'{i+1:03d};{value}')
            data.write(f'{value} \n')

async def display_sensor(count = 1):
    # Read sensor value and print to console
    # Checked 2025-05-20/Kay Struebing on Base Tester
    sensor = setup_serial_mitutoyo()
    for i in range(count):
        value = read_value(sensor)
        print(f'{i+1:03d}: {value}')

async def zero_test():
    # Test the zero function of the Mitutoyo sensor
    # Checked 2025-05-20/Kay Struebing on Base Tester
    sensor = setup_serial_mitutoyo()
    value1 = read_value(sensor)
    response = set_zero(sensor)
    value2 = read_value(sensor)
    print(f'{value1}, {response}, {value2}')
    return


async def init(pos_mode=2, pos_stable_time=100, pos_range=1):
    await start_firmware('eef')
    await st_unit.InitializeDevice()
    await st_endpoint.SetParameter(DualMoverParameter.PositioningModeA, pos_mode)
    await st_endpoint.SetParameter(DualMoverParameter.PositioningModeB, pos_mode)
    await st_endpoint.SetParameter(DualMoverParameter.PositioningStableTimeA, pos_stable_time)
    await st_endpoint.SetParameter(DualMoverParameter.PositioningStableTimeB, pos_stable_time)
    await st_endpoint.SetParameter(DualMoverParameter.PositioningRangeA, pos_range)
    await st_endpoint.SetParameter(DualMoverParameter.PositioningRangeB, pos_range)
    await st_endpoint.SetParameter(DualMoverParameter.MaxEncoderDeviationA, 0xFFFFFFFF)
    await st_endpoint.SetParameter(DualMoverParameter.MaxEncoderDeviationB, 0xFFFFFFFF)


async def move(position_x, position_y, log_interval=0.05, extra_log_time=0.5):
    done = False

    async def move_task(position_x, position_y, extra_log_time):
        nonlocal done, start_time
        await st_endpoint.Move(position_x, position_y)
        PyLogger.logger.info(f"; {time.perf_counter() - start_time:6.3f} ; mover stopped")
        await asyncio.sleep(extra_log_time)
        done = True

    async def log_task(interval):
        nonlocal done, start_time
        PyLogger.logger.info(f"; time   ; enc_x  ; enc_y")
        start_time = time.perf_counter()
        while not done:
            await asyncio.sleep(interval)
            enc_x, enc_y = await st_endpoint.GetEncoderPosition()
            PyLogger.logger.info(f"; {time.perf_counter() - start_time:6.3f} ; {enc_x:6d} ; {enc_y:6d}")

    start_time = time.perf_counter()
    await asyncio.gather(
        move_task(position_x, position_y, extra_log_time),
        log_task(log_interval),
    )
    await asyncio.sleep(0.1)
    pos_x, pos_y = await st_endpoint.GetPosition()
    enc_x, enc_y = await st_endpoint.GetEncoderPosition()
    PyLogger.logger.info(f"  Mover Position: {pos_x:6d}, {pos_y:6d}")
    PyLogger.logger.info(f"Encoder Position: {enc_x:6d}, {enc_y:6d}")
    await asyncio.sleep(0.1)


async def encoder_monitor(duration=60.0, interval=1.0):
    await start_firmware('eef')
    await st_endpoint.SetParameter(DualMoverParameter.EncoderCorrectionFactorA, 1.0)
    await st_endpoint.SetParameter(DualMoverParameter.EncoderCorrectionFactorB, 1.0)
    await st_endpoint.SetParameter(DualMoverParameter.EncoderModeA, 1)
    await st_endpoint.SetParameter(DualMoverParameter.EncoderModeB, 1)

    for _ in range(round(duration / interval)):
        await asyncio.sleep(interval)
        enc_x, enc_y = await st_endpoint.GetEncoderPosition()
        PyLogger.logger.info(f"Encoder Position: {enc_x}, {enc_y}")

# Mechanical Test with Mitutoyo Gauge
# https://revvitydev.atlassian.net/wiki/x/GYCIYQ
# 2025-05-22 Kay Struebing

async def log_position(mode = 0, direction = 'y', profile = 0, y_range: float = 8.0, step = 0.1, delay = 0.2):
    # Moves scantable for range mm in direction (positive) and logs position from mitutoyo-sensor

    # !! Scantable has to be positioned before starting the script !!
    # direction: Movement direction to test: 'x'; 'y'
    # profile: Motor profile for movement
    # range. mm to move 
    # step: Step of movement in mm
    # delay: Delay between steps in seconds

    test = sys._getframe().f_code.co_name  # this function

    # Write results to file:
    data_filename = get_datafile(test)

    
    # Initialize serial port for Mitutoyo sensor
    taster = setup_serial_mitutoyo()
        
    # Initialize scantable
    await init(mode, 100, 1)
    
    # Set profile
    await st_unit.UseProfile(profile)
    
    # Start at current position (sensor in place!)
    x, y = await st_unit.GetPosition()
    time.sleep(delay)
    no_steps = int(y_range / step)

    try:

        with open (data_filename, 'a') as data:
            data.write(f'Result of {test}')
            data.write(f'mode = {mode}, direction = {direction}, profile = {profile}, delay = {delay}\n')
            # First Position Value
            data.write(f'no;st_x;st_y;pos;enc_x;enc_y\n')
            response = set_zero(taster)
            print(f'Taster set to zero {response}')
            taster_value = read_value(taster)
            enc_x, enc_y = await st_endpoint.GetEncoderPosition()
            data.write(f'0000;{x};{y};{taster_value};{enc_x};{enc_y}\n')
            time.sleep(delay)
            # Move in X/Y stepwise
            for nr in range(no_steps):
                if direction == 'y':
                    y += step
                elif direction == 'x':
                    x += step
                else:
                    return('stopped by user')
                await st_unit.Move(x, y)
                time.sleep(delay)
                taster_value = read_value(taster)
                enc_x, enc_y = await st_endpoint.GetEncoderPosition()
                record = f'{(nr+1):04d};{x:7.3f};{y:7.3f};{taster_value};{enc_x};{enc_y}\n'
                data.write(record)
                print(record)
            for nr in range(no_steps, 0, -1):
                if direction == 'y':
                    y -= step
                elif direction == 'x':
                    x -= step
                else:
                    return('stopped by user')
                await st_unit.Move(x, y)
                time.sleep(delay)
                taster_value = read_value(taster)
                enc_x, enc_y = await st_endpoint.GetEncoderPosition()
                record = f'{nr:04d};{x:7.3f};{y:7.3f};{taster_value};{enc_x};{enc_y}\n'
                data.write(record)
                print(record)

    except Exception as ex:
        print(f"something's going wrong this way: {ex}") 
        return(ex)   
    
    print(f'Testscript {test} finished with no errors, result written to {data_filename}')
    return(data_filename)

async def repeat_position(mode = 0, repeats = 10, distance = 10, direction = 'y', profile = 0, delay = 0.2):
    # Move from current Position to fixed distance and measure the position
    test = sys._getframe().f_code.co_name  # this function

    # Write results to file:
    data_filename = get_datafile(test)

    
    # Initialize serial port for Mitutoyo sensor
    taster = setup_serial_mitutoyo()
        
    # Initialize scantable
    await init(mode, 100, 1)
    
    # Set profile
    await st_unit.UseProfile(profile)
    
    # Start at current position (sensor in place!)
    start_x, start_y = await st_unit.GetPosition()
    if start_x == None:
        return('Position not initialized')
    
    if direction == 'y':
        x = start_x
        y = start_y + distance
    elif direction == 'x':
        x = start_x + distance
        y = start_y
    else:
        return('stopped by user')

    try:

        with open (data_filename, 'a') as data:
            data.write(f'Result of {test}')
            data.write(f'mode = {mode}, distance = {distance}, direction = {direction}, profile = {profile}, delay = {delay}\n')
            # First Position Value
            data.write(f'no;st_x;st_y;pos;enc_x;enc_y\n')
            response = set_zero(taster)
            print(f'Taster set to zero {response}')
            taster_value = read_value(taster)
            enc_x, enc_y = await st_endpoint.GetEncoderPosition()
            data.write(f'0000;{x};{y};{taster_value};{enc_x};{enc_y}\n')
            time.sleep(delay)
            # Move to target
            for nr in range(repeats):
                await st_unit.Move(x, y)
                time.sleep(delay)
                taster_value = read_value(taster)
                enc_x, enc_y = await st_endpoint.GetEncoderPosition()
                record = f'{nr:04d};{x:7.3f};{y:7.3f};{taster_value};{enc_x};{enc_y}\n'
                data.write(record)
                print(record)
                await st_unit.Move(start_x, start_y)
                time.sleep(delay)
                taster_value = read_value(taster)
                enc_x, enc_y = await st_endpoint.GetEncoderPosition()
                record = f'{nr:04d};{x:7.3f};{y:7.3f};{taster_value};{enc_x};{enc_y}\n'
                data.write(record)
                print(record)


    except Exception as ex:
        print(f"something's going wrong this way: {ex}") 
        return(ex)   
    
    print(f'Testscript {test} finished with no errors, result written to {data_filename}')
    return(data_filename)

